<?php

use Elementor\Utils;
use Elementor\Embed;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class LQD_Vertical_Slideshow extends Widget_Base {

	public function get_name() {
		return 'lqd-vertical-slideshow';
	}

	public function get_title() {
		return __( 'Liquid Vertical Slideshow', 'logistics-hub-core' );
	}

	public function get_icon() {
		return 'eicon-slider-push lqd-element';
	}

	public function get_categories() {
		return [ 'liquid-core' ];
	}

	public function get_keywords() {
		return [ 'slideshow', 'carousel', 'slider' ];
	}

	public function get_behavior() {
		$settings = $this->get_settings_for_display();
		$carousel_options = [];

		if ( !empty( $settings['friction']['size'] ) && $settings['friction']['size'] !== 0.28 ) {
			$carousel_options['friction'] = $settings['friction']['size'];
		}
		if ( !empty( $settings['selected_attraction']['size'] ) && $settings['selected_attraction']['size'] !== 0.28 ) {
			$carousel_options['selectedAttraction'] = $settings['selected_attraction']['size'];
		}
		if ( $settings['wrap_around'] === 'yes' ) {
			$carousel_options['wrapAround'] = true;
		}
		if ( $settings['group_cells'] === 'yes' ) {
			$carousel_options['groupItems'] = true;
		}

		$behavior = [
			[
				'behaviorClass' => 'LiquidCarouselBehavior',
				'options' => $carousel_options
			]
		];

		if ( $settings['draggable'] === 'yes' ) {
			$draggable_options = [];

			if ( $settings['free_scroll'] === 'yes' ) {
				$draggable_options['freeScroll'] = true;
			}
			if ( !empty( $settings['free_scroll_friction']['size'] ) && $settings['free_scroll_friction']['size'] != 0.075 ) {
				$draggable_options['freeScrollFriction'] = $settings['free_scroll_friction']['size'];
			}

			$behavior[] = [
				'behaviorClass' => 'LiquidCarouselDragBehavior',
				'options' => $draggable_options
			];
		}
		if ( $settings['nav_buttons'] === 'yes' ) {
			$behavior[] = [
				'behaviorClass' => 'LiquidCarouselNavBehavior',
			];
		}
		if ( $settings['pagination_dots'] === 'yes' ) {
			$behavior[] = [
				'behaviorClass' => 'LiquidCarouselDotsBehavior',
			];
		}
		if ( $settings['autoplay_time'] > 0 ) {
			$autoplay_options = [
				'autoplayTimeout' => $settings['autoplay_time']
			];

			if ( $settings['pause_autoplay_onhover'] === 'yes' ) {
				$autoplay_options['pauseAutoPlayOnHover'] = true;
			}

			$behavior[] = [
				'behaviorClass' => 'LiquidCarouselAutoplayBehavior',
				'options' => $autoplay_options
			];
		}
		if ( $settings['slides_indicator'] === 'yes' ) {
			$behavior[] = [
				'behaviorClass' => 'LiquidCarouselSlidesIndicatorBehavior',
				'options' => [
					'leadingZero' => $settings['slides_indicator_numbers_leading_zero'] === 'yes'
				]
			];
		}

		foreach ( $settings['cells'] as $i => $cell ) {
			if ( $cell['background_background'] === 'video' ) {

				$videoUrl = $cell[ 'background_video_link' ];


				if ( empty( $videoUrl ) ) continue;

				$cellId = $cell[ '_id' ];
				$start = $cell[ 'background_video_start' ];
				$end = $cell[ 'background_video_end' ];
				$playOnce = $cell[ 'background_play_once' ];
				$playOnMobile = $cell[ 'background_play_on_mobile' ];
				$privacyMode = $cell[ 'background_privacy_mode' ];

				$behavior[] = [
					'behaviorClass' => 'LiquidVideoBackgroundBehavior',
					'options' => [
						'videoUrl' => "'$videoUrl'",
						'start' => $start !== '' ? $start  : null,
						'end' => $end !== '' ? $end  : null,
						'playOnce' => $playOnce === 'yes',
						'privacyMode' => $privacyMode === 'yes',
						'playOnMobile' => $playOnMobile === 'yes',
						'ui' => [
							'videoContainer' => "'.lqd-bg-video-container-$cellId'"
						]
					]
				];

			}
		}

		return $behavior;
	}

	protected function register_controls() {

		$this->start_controls_section(
			'general_section',
			[
				'label' => __( 'General', 'logistics-hub-core' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'title', [
				'title' => esc_html__( 'Title', 'logistics-hub-core' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Cost-effective Solutions' , 'logistics-hub-core' ),
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'description',
			[
				'label' => esc_html__( 'Description', 'logistics-hub-core' ),
				'type' => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed. ', 'logistics-hub-core' ),
			]
		);

		$repeater->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'background',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} {{CURRENT_ITEM}} .lqd-v-slideshow-image',
			]
		);

		$this->add_control(
			'cells',
			[
				'label' => esc_html__( 'Items', 'logistics-hub-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'title' => 'Reliable Schedules',
						'description' => 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed.',
					],
					[
						'title' => 'Cost-effective Solutions',
						'description' => 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed.',
					],
					[
						'title' => 'High Cargo Capacity',
						'description' => 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed.',
					],
					[
						'title' => 'Long-Term Contracts',
						'description' => 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed.',
					],
					[
						'title' => 'Trusted experience',
						'description' => 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed.',
					],
					[
						'title' => 'Easy Freight Booking',
						'description' => 'At SwiftTrans, we are passionate about providing efficient and reliable logistics solutions that help our customers succeed.',
					],
				],
				'title_field' => '{{{ title }}}',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label' => esc_html__( 'Title HTML Tag', 'logistics-hub-core' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'p' => 'p',
					'div' => 'div',
					'span' => 'span',
				],
				'default' => 'h3',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'friction',
			[
				'label' => esc_html__( 'Friction', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'description' => esc_html__( 'Default is 0.28', 'logistics-hub-core' ),
				'size_units' => [ 'px' ],
				'default' => [
					'unit' => 'px',
					'size' => 0.28
				],
				'range' => [
					'px' => [
						'min' => 0.01,
						'max' => 2,
						'step' => 0.01
					]
				],
				'render_type' => 'template',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'selected_attraction',
			[
				'label' => esc_html__( 'Selected attraction', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'description' => esc_html__( 'Default is 0.025', 'logistics-hub-core' ),
				'size_units' => [ 'px' ],
				'default' => [
					'unit' => 'px',
					'size' => 0.025
				],
				'range' => [
					'px' => [
						'min' => 0.01,
						'max' => 2,
						'step' => 0.001
					]
				],
				'render_type' => 'template'
			]
		);

		$this->add_control(
			'wrap_around',
			[
				'label' => __( 'Loop', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes'
			]
		);

		$this->add_control(
			'group_cells',
			[
				'label' => __( 'Group cells', 'logistics-hub-core' ),
				'description' => __( 'Enable this option if you want the navigation being mapped to grouped cells, not individual cells.', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'draggable',
			[
				'label' => __( 'Draggable', 'logistics-hub-core' ),
				'description' => __( 'Enable/Disable draggableity of the carousel.', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes'
			]
		);

		$this->add_control(
			'free_scroll',
			[
				'label' => __( 'Free scroll', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'draggable' => 'yes'
				]
			]
		);

		$this->add_control(
			'free_scroll_friction',
			[
				'label' => esc_html__( 'Free scroll friction', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'description' => esc_html__( 'Default is 0.075', 'logistics-hub-core' ),
				'size_units' => [ 'px' ],
				'default' => [
					'unit' => 'px',
					'size' => 0.075
				],
				'range' => [
					'px' => [
						'min' => 0.001,
						'max' => 2,
						'step' => 0.01
					]
				],
				'render_type' => 'template',
				'condition' => [
					'draggable' => 'yes',
					'free_scroll' => 'yes'
				]
			]
		);

		$this->add_control(
			'autoplay_time',
			[
				'label' => __( 'Autoplay delay', 'logistics-hub-core' ),
				'type' => Controls_Manager::NUMBER,
				'description' => __( 'Set a number if you want to enable autoplay. Number should be in <strong>milliseconds</strong>.', 'logistics-hub-core' ),
				'placeholder' => '0',
				'step' => 1000,
			]
		);

		$this->add_control(
			'pause_autoplay_onhover',
			[
				'label' => __( 'Pause autoplay on hover', 'logistics-hub-core' ),
				'description' => __( 'Pause the autoplay each time user hovers over the carousel.', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'autoplay_time!' => '',
				]
			]
		);

		$this->add_control(
			'nav_buttons',
			[
				'label' => __( 'Navigation buttons', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'pagination_dots',
			[
				'label' => __( 'Pagination dots', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'slides_indicator',
			[
				'label' => __( 'Slides indicator', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'positioning_alignment_section',
			[
				'label' => __( 'Positioning & Alignments', 'logistics-hub-core' ),
			]
		);

		$this->add_responsive_control(
			'cell_text_align',
			[
				'label' => __( 'Text align', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => __( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'end' => [
						'title' => __( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-v-slideshow-text' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'align_content',
			[
				'label' => esc_html__( 'Align content', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-align-start-v',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-align-center-v',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-align-end-v',
					],
					'' => [
						'title' => esc_html__( 'Stretch', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-align-stretch-v',
					],
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-cell-content' => 'align-items: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		// Navigation Section
		$this->start_controls_section(
			'navigation_section',
			[
				'label' => __( 'Navigation', 'logistics-hub-core' ),
				'condition' => [
					'nav_buttons' => 'yes'
				]
			]
		);

		$this->add_control(
			'nav_arrows_style',
			[
				'label' => __( 'Style', 'logistics-hub-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1' => __( 'Style 1', 'logistics-hub-core' ),
					'custom' => __( 'Custom', 'logistics-hub-core' ),
				],
			]
		);

		$this->add_control(
			'nav_prev_icon',
			[
				'label' => esc_html__( 'Previous icon', 'logistics-hub-core' ),
				'type' => Controls_Manager::ICONS,
				'skin' => 'inline',
				'default' => [
					'value' => 'fas fa-arrow-left',
					'library' => 'fa-solid',
				],
				'condition' => [
					'nav_arrows_style' => 'custom'
				]
			]
		);

		$this->add_control(
			'nav_next_icon',
			[
				'label' => esc_html__( 'Next icon', 'logistics-hub-core' ),
				'type' => Controls_Manager::ICONS,
				'skin' => 'inline',
				'default' => [
					'value' => 'fas fa-arrow-right',
					'library' => 'fa-solid',
				],
				'condition' => [
					'nav_arrows_style' => 'custom'
				]
			]
		);

		$this->add_responsive_control(
			'nav_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'outside' => [
						'title' => esc_html__( 'Outside', 'logistics-hub-core' ),
						'icon' => 'eicon-sign-out',
					],
					'inside' => [
						'title' => esc_html__( 'Inside', 'logistics-hub-core' ),
						'icon' => 'eicon-square',
					],
				],
				'default' => 'inside',
				'toggle' => false,
				'selectors_dictionary' => [
					'outside' => 'relative',
					'inside' => 'absolute'
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'position: {{VALUE}}'
				]
			]
		);

		$this->add_responsive_control(
			'nav_align',
			[
				'label' => __( 'Alignment', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
					'space-between' => [
						'title' => esc_html__( 'Space between', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-stretch',
					],
				],
				'default' => 'start',
				'toggle' => false,
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'justify-content: {{VALUE}};'
				],
				'condition' => [
					'nav_separate_buttons_offset' => ''
				],
			]
		);

		$this->add_responsive_control(
			'nav_orientation_h',
			[
				'label' => esc_html__( 'Horizontal orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => 'px',
					'size' => 30
				],
				'condition' => [
					'nav_orientation_h' => 'start'
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'nav_orientation_h' => 'end'
				],
			]
		);

		$this->add_responsive_control(
			'nav_orientation_v',
			[
				'label' => esc_html__( 'Vertical orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'top: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => 'px',
					'size' => 30
				],
				'condition' => [
					'nav_orientation_v' => 'top'
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'nav_orientation_v' => 'bottom'
				],
			]
		);

		$this->add_responsive_control(
			'nav_rotate',
			[
				'label' => esc_html__( 'Rotation', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => -360,
						'max' => 360,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav' => 'width: auto; transform: rotate({{SIZE}}deg)',
				],
			]
		);

		$this->add_control(
			'nav_separate_buttons_offset',
			[
				'label' => __( 'Offset buttons separately', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
				]
			]
		);

		$this->add_responsive_control(
			'prev_button_orientation_h',
			[
				'label' => esc_html__( 'Prev button Horizontal orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-prev' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-prev' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'prev_button_orientation_h' => 'start',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-prev' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'prev_button_orientation_h' => 'end',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_orientation_v',
			[
				'label' => esc_html__( 'Prev button vertical orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-prev' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-prev' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'prev_button_orientation_v' => 'top',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-prev' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'prev_button_orientation_v' => 'bottom',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_orientation_h',
			[
				'label' => esc_html__( 'Next button horizontal orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'end',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-next' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-next' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'next_button_orientation_h' => 'start',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-next' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'next_button_orientation_h' => 'end',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_orientation_v',
			[
				'label' => esc_html__( 'Next button vertical orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-next' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-next' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'next_button_orientation_v' => 'top',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-nav-next' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'next_button_orientation_v' => 'bottom',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_control(
			'nav_appear_onhover',
			[
				'label' => __( 'Appear on hover', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'condition' => [
					'nav_placement' => 'inside'
				]
			]
		);

		$this->end_controls_section();

		// Pagination Dots
		$this->start_controls_section(
			'pagination_dots_section',
			[
				'label' => __( 'Pagination dots', 'logistics-hub-core' ),
				'condition' => [
					'pagination_dots' => 'yes'
				]
			]
		);

		$this->add_control(
			'dots_style',
			[
				'label' => __( 'Style', 'logistics-hub-core' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'1' => __( 'Style 1', 'logistics-hub-core' ),
				],
				'default' => '1'
			]
		);

		$this->add_responsive_control(
			'dots_orientation',
			[
				'label' => esc_html__('Orientation', 'logistics-hub-core'),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'' => [
						'title' => esc_html__('Horizontal', 'logistics-hub-core'),
						'icon' => 'eicon-ellipsis-h',
					],
					'column' => [
						'title' => esc_html__('Vertical', 'logistics-hub-core'),
						'icon' => 'eicon-ellipsis-v',
					],
				],
				'default' => '',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'flex-direction: {{VALUE}}'
				]
			]
		);

		$this->add_responsive_control(
			'dots_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'outside' => [
						'title' => esc_html__( 'Outside', 'logistics-hub-core' ),
						'icon' => 'eicon-sign-out',
					],
					'inside' => [
						'title' => esc_html__( 'Inside', 'logistics-hub-core' ),
						'icon' => 'eicon-square',
					],
				],
				'default' => 'outside',
				'toggle' => false,
				'selectors_dictionary' => [
					'outside' => 'relative',
					'inside' => 'absolute',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'position: {{VALUE}};'
				]
			]
		);

		$this->add_responsive_control(
			'dots_align',
			[
				'label' => __( 'Alignment', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
					'space-between' => [
						'title' => esc_html__( 'Space between', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-stretch',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'justify-content: {{VALUE}};'
				],
				'condition' => [
					'dots_placement' => 'outside'
				]
			]
		);

		$this->add_responsive_control(
			'dots_placement_h',
			[
				'label' => esc_html__( 'Horizontal placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_h' => 'start'
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_h' => 'end'
				],
			]
		);

		$this->add_responsive_control(
			'dots_placement_v',
			[
				'label' => esc_html__( 'Vertical placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_v' => 'top'
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_v' => 'bottom'
				],
			]
		);

		$this->add_responsive_control(
			'dots_rotate',
			[
				'label' => esc_html__( 'Rotation', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => -360,
						'max' => 360,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-dots' => 'width: auto; transform: rotate({{SIZE}}deg)',
				],
			]
		);

		$this->end_controls_section();

		// Slides Indicator Section
		$this->start_controls_section(
			'slides_indicator_section',
			[
				'label' => __( 'Slides indicator', 'logistics-hub-core' ),
				'condition' => [
					'slides_indicator' => 'yes'
				]
			]
		);

		$this->add_control(
			'slides_indicator_numbers',
			[
				'label' => __( 'Show current & total numbers?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'condition' => [
					'slides_indicator' => 'yes'
				]
			]
		);

		$this->add_control(
			'slides_indicator_numbers_leading_zero',
			[
				'label' => __( 'Numbers leading zero?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_numbers' => 'yes',
				]
			]
		);

		$this->add_control(
			'slides_indicator_tail',
			[
				'label' => __( 'Process tail?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'condition' => [
					'slides_indicator' => 'yes'
				]
			]
		);

		$this->add_control(
			'slides_indicator_inline_w_nav',
			[
				'label' => __( 'Inline with navigation?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'slides_indicator' => 'yes',
					'nav_buttons' => 'yes',
					'slides_indicator_inline_w_dots' => '',
				]
			]
		);

		$this->add_responsive_control(
			'slides_indicator_inline_w_nav_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'condition' => [
					'slides_indicator' => 'yes',
					'nav_buttons' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => 'yes',
				]
			]
		);

		$this->add_control(
			'slides_indicator_inline_w_dots',
			[
				'label' => __( 'Inline with dots?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'slides_indicator' => 'yes',
					'pagination_dots' => 'yes',
					'slides_indicator_inline_w_nav' => '',
				]
			]
		);

		$this->add_responsive_control(
			'slides_indicator_inline_w_dots_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => 'yes',
					'slides_indicator_inline_w_nav' => '',
				]
			]
		);

		$this->add_control(
			'slides_indicator_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'outside' => [
						'title' => esc_html__( 'Outside', 'logistics-hub-core' ),
						'icon' => 'eicon-sign-out',
					],
					'inside' => [
						'title' => esc_html__( 'Inside', 'logistics-hub-core' ),
						'icon' => 'eicon-square',
					],
				],
				'default' => 'outside',
				'toggle' => false,
				'condition' => [
					'slides_indicator' => 'yes',
					'pagination_dots' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
				]
			]
		);

		$this->add_responsive_control(
			'slides_indicator_align',
			[
				'label' => __( 'Alignment', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
					'space-between' => [
						'title' => esc_html__( 'Space between', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-stretch',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-slides-ind' => 'justify-content: {{VALUE}};'
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'outside'
				]
			]
		);

		$this->add_control(
			'slides_indicator_placement_h',
			[
				'label' => esc_html__( 'Horizontal placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-slides-ind' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_h' => 'start'
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-slides-ind' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_h' => 'end'
				],
			]
		);

		$this->add_control(
			'slides_indicator_placement_v',
			[
				'label' => esc_html__( 'Vertical placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-slides-ind' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_v' => 'top'
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-slides-ind' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_v' => 'bottom'
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_rotate',
			[
				'label' => esc_html__( 'Rotation', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => -360,
						'max' => 360,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-carousel-slides-ind' => 'width: auto; transform: rotate({{SIZE}}deg)',
				],
			]
		);

		$this->end_controls_section();

		\LQD_Elementor_Helper::add_style_controls(
			$this,
			'carousel',
			[
				'cell' => [
					'controls' => [
						[
							'type' => 'width',
							'css_var' => '--lqd-carousel-cell-w',
							'default' => [
								'unit' => '%',
								'size' => 25
							]
						],
						[
							'type' => 'height',
							'css_var' => '--lqd-carousel-cell-h',
							'default' => [
								'unit' => 'vh',
								'size' => 50
							]
						],
						[
							'type' => 'padding',
							'css_var' => '--lqd-carousel-cell-p'
						],
						[
							'type' => 'liquid_background_css',
							'name' => 'overlay_bg',
							'css_var' => '--lqd-carousel-cell-overlay-bg',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Overlay background', 'logistics-hub-core' ),
								],
							],
						],
						[
							'type' => 'border',
							'css_var' => '--lqd-carousel-cell-br'
						],
						[
							'type' => 'border_radius',
						],
						[
							'type' => 'box_shadow',
							'css_var' => '--lqd-carousel-cell-bs'
						],
					],
					'state_tabs' => [ 'normal', 'hover' ],
				],
				'cell_text' => [
					'label' => 'Cell content',
					'controls' => [
						[
							'type' => 'margin',
							'css_var' => '--lqd-v-slideshow-text-m'
						],
						[
							'type' => 'padding',
							'css_var' => '--lqd-v-slideshow-text-p'
						],
						[
							'type' => 'opacity',
							'selector' => '.lqd-v-slideshow-text'
						],
						[
							'type' => 'liquid_background_css',
							'css_var' => '--lqd-v-slideshow-text-bg',
						],
						[
							'type' => 'border',
							'css_var' => '--lqd-v-slideshow-text-br'
						],
						[
							'type' => 'border_radius',
							'selector' => '.lqd-v-slideshow-text'
						],
						[
							'type' => 'box_shadow',
							'css_var' => '--lqd-v-slideshow-text-bs'
						],
					],
					'state_tabs' => [ 'normal', 'hover' ],
					'state_selectors_before' => [ 'hover' => ' .lqd-carousel-cell' ]
				],
				'cell_title' => [
					'controls' => [
						[
							'type' => 'typography',
						],
						[
							'type' => 'margin',
						],
						[
							'type' => 'padding',
						],
						[
							'type' => 'opacity',
						],
						[
							'type' => 'liquid_color',
						],
						[
							'type' => 'liquid_background_css',
						],
						[
							'type' => 'border',
						],
						[
							'type' => 'border_radius',
						],
						[
							'type' => 'box_shadow',
						],
					],
					'state_tabs' => [ 'normal', 'hover'],
					'state_selectors_before' => [ 'hover' => ' .lqd-carousel-cell' ]
				],
				'cell_description' => [
					'controls' => [
						[
							'type' => 'typography',
						],
						[
							'type' => 'margin',
						],
						[
							'type' => 'padding',
						],
						[
							'type' => 'opacity',
						],
						[
							'type' => 'liquid_color',
						],
						[
							'type' => 'liquid_background_css',
						],
						[
							'type' => 'border',
						],
						[
							'type' => 'border_radius',
						],
						[
							'type' => 'box_shadow',
						],
					],
					'state_tabs' => [ 'normal', 'hover' ],
					'state_selectors_before' => [ 'hover' => ' .lqd-carousel-cell' ]
				],
				'nav' => [
					'label' => 'Navigation button',
					'controls' => [
						[
							'type' => 'font_size',
							'label' => 'Icon size',
							'css_var' => '--lqd-carousel-nav-icon-size'
						],
						[
							'type' => 'gap',
							'label' => 'Gap between buttons',
							'css_var' => '--lqd-carousel-nav-gap'
						],
						[
							'type' => 'liquid_linked_dimensions',
							'css_var' => '--lqd-carousel-nav-btn'
						],
						[
							'type' => 'margin',
							'css_var' => '--lqd-carousel-nav-m',
						],
						[
							'type' => 'padding',
							'css_var' => '--lqd-carousel-nav-p'
						],
						[
							'type' => 'liquid_color',
							'css_var' => '--lqd-carousel-nav-btn-color',
						],
						[
							'type' => 'liquid_background_css',
							'css_var' => '--lqd-carousel-nav-btn-bg',
						],
						[
							'type' => 'border',
							'css_var' => '--lqd-carousel-nav-btn-br'
						],
						[
							'type' => 'border_radius',
							'css_var' => '--lqd-carousel-nav-btn-brr'
						],
						[
							'type' => 'box_shadow',
							'css_var' => '--lqd-carousel-nav-btn-bs'
						],
					],
					'condition' => [
						'nav_buttons' => 'yes'
					],
					'state_tabs' => [ 'normal', 'hover' ],
					'state_selectors' => [ 'hover' => ' .lqd-carousel-nav-btn:not([disabled]):hover' ]
				],
				'dots' => [
					'label' => 'Pagination dot',
					'controls' => [
						[
							'type' => 'gap',
							'label' => 'Gap between dots',
							'css_var' => '--lqd-carousel-dots-gap'
						],
						[
							'type' => 'liquid_linked_dimensions',
							'css_var' => '--lqd-carousel-dots'
						],
						[
							'type' => 'margin',
							'css_var' => '--lqd-carousel-dots-m'
						],
						[
							'type' => 'liquid_background_css',
							'css_var' => '--lqd-carousel-dots-bg',
						],
						[
							'type' => 'border',
							'css_var' => '--lqd-carousel-dots-br'
						],
						[
							'type' => 'border_radius',
							'css_var' => '--lqd-carousel-dots-brr',
							'selector' => '.lqd-carousel-dots .lqd-carousel-dot'
						],
						[
							'type' => 'box_shadow',
							'css_var' => '--lqd-carousel-dots-bs'
						],
					],
					'condition' => [
						'pagination_dots' => 'yes'
					],
					'state_tabs' => [ 'normal', 'hover', 'active' ],
					'state_selectors' => [ 'hover' => ' .lqd-carousel-dot:hover', 'active' => ' .lqd-carousel-dot-active' ]
				],
				'slides_indicator' => [
					'controls' => [
						[
							'type' => 'typography',
						],
						[
							'type' => 'gap',
						],
						[
							'type' => 'width',
							'label' => 'Indicator width',
							'css_var' => '--lqd-carousel-slide-progress-w'
						],
						[
							'type' => 'height',
							'label' => 'Indicator height',
							'css_var' => '--lqd-carousel-slide-progress-h'
						],
						[
							'type' => 'margin',
							'css_var' => '--lqd-carousel-slide-progress-m',
						],
						[
							'type' => 'padding',
							'css_var' => '--lqd-carousel-slide-progress-p'
						],
						[
							'type' => 'liquid_background_css',
							'name' => 'slides_indicator_bar_wrap_bg',
							'label' => 'Indicator background',
							'css_var' => '--lqd-carousel-slide-bar-wrap-bg',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Bar background', 'logistics-hub-core' ),
								],
							],
						],
						[
							'type' => 'liquid_background_css',
							'name' => 'slides_indicator_bar_bg',
							'label' => 'Indicator bar background',
							'css_var' => '--lqd-carousel-slide-bar-bg',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Filled bar background', 'logistics-hub-core' ),
								],
							],
						],
						[
							'type' => 'liquid_color',
							'name' => 'slides_indicator_color',
							'label' => 'Numbers color'
						],
					],
					'selector' => '.lqd-carousel-slides-ind',
					'condition' => [
						'slides_indicator' => 'yes'
					]
				]
			],
		);

	}

	protected function add_render_attributes() {

		parent::add_render_attributes();

		$wrapper_classnames[] = 'lqd-group-carousel';

		if ( empty( $this->get_settings_for_display( '_element_width' ) ) ) {
			$wrapper_classnames[] = 'w-full';
		}

		if ( $this->get_settings_for_display( 'draggable' ) === 'yes' ) {
			$wrapper_classnames[] = 'lqd-carousel-is-draggable';
		}

		$this->add_render_attribute( '_wrapper', [
			'class' => $wrapper_classnames
		] );

	}

	protected function get_nav_icons_style_custom() {
		$settings = $this->get_settings_for_display();

		$arrows = [
			'prev' => '',
			'next' => ''
		];

		if ( !empty( $settings['nav_prev_icon']['value'] ) ) {
			$arrows['prev'] = Icons_Manager::try_get_icon_html( $settings['nav_prev_icon'], [ 'aria-hidden' => 'true', 'class' => 'w-1em h-auto align-middle fill-current relative' ] );
		}
		if ( !empty( $settings['nav_next_icon']['value'] ) ) {
			$arrows['next'] = Icons_Manager::try_get_icon_html( $settings['nav_next_icon'], [ 'aria-hidden' => 'true', 'class' => 'w-1em h-auto align-middle fill-current relative' ] );
		}

		return $arrows;
	}

	protected function get_nav_icons_style_1() {
		$settings = $this->get_settings_for_display();

		return [
			'prev' => '<svg width="16" height="13" viewBox="0 0 16 13" fill="currentColor" xmlns="http://www.w3.org/2000/svg"><path d="M6.029 -9.53674e-07L6.972 0.942999L2.522 5.393L15.688 5.393V6.706L2.522 6.706L6.972 11.156L6.029 12.099L0.451004 6.525L3.8147e-06 6.053L0.451004 5.581L6.029 -9.53674e-07Z"/></svg>',
			'next' => '<svg width="16" height="13" viewBox="0 0 16 13" fill="currentColor" xmlns="http://www.w3.org/2000/svg"><path d="M9.659 0L8.716 0.943L13.166 5.393H0V6.706H13.166L8.716 11.156L9.659 12.099L15.237 6.525L15.688 6.053L15.237 5.581L9.659 0Z"/></svg>'
		];
	}

	protected function get_nav() {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['nav_buttons'] ) ) return '';

		$nav_classnames = [ 'lqd-carousel-nav', 'flex', 'items-center', 'w-full', 'pointer-events-none' ];
		$buttons_common_classnames = [ 'lqd-carousel-nav-btn', 'inline-flex', 'items-center', 'justify-center', 'pointer-events-auto', 'transition-all', 'cursor-pointer' ];
		$nav_style = $settings['nav_arrows_style'];
		$slides_indicator = $settings['slides_indicator'];
		$slides_indicator_is_inline = $settings['slides_indicator_inline_w_nav'];
		$slides_indicator_placement = $settings['slides_indicator_inline_w_nav_placement'];
		$should_get_slides_indicator = $slides_indicator === 'yes' && $slides_indicator_is_inline === 'yes';

		if ( $settings['nav_separate_buttons_offset'] === 'yes' ) {
			$buttons_common_classnames[] = 'absolute';
		}
		if ( $settings['nav_appear_onhover'] === 'yes' ) {
			$buttons_common_classnames[] = 'opacity-0';
			$buttons_common_classnames[] = 'lqd-group-carousel-hover:opacity-100';
		}

		$this->add_render_attribute( 'carousel_nav_prev_btn', [
			'class' => array_merge(
				['lqd-carousel-nav-prev'],
				$buttons_common_classnames
			),
			'type' => 'button',
			'aria-label' => esc_attr__('Previous', 'logistics-hub-core'),
			'disabled' => true
		] );
		$this->add_render_attribute( 'carousel_nav_next_btn', [
			'class' => array_merge(
				['lqd-carousel-nav-next'],
				$buttons_common_classnames
			),
			'type' => 'button',
			'aria-label' => esc_attr__('Next', 'logistics-hub-core'),
			'disabled' => true
		] );
		$this->add_render_attribute( 'carousel_nav', [
			'class' => $nav_classnames
		] );

		$nav_icons = $this->{'get_nav_icons_style_' . $nav_style}( $settings );

		?><div <?php $this->print_render_attribute_string( 'carousel_nav' ) ?>>
			<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'start' ) {
				$this->get_slides_indicator( $settings );
			} ?>
			<button <?php $this->print_render_attribute_string('carousel_nav_prev_btn'); ?>><?php
				echo $nav_icons['prev'];
			?></button>
			<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'center' ) {
				$this->get_slides_indicator( $settings );
			} ?>
			<button <?php $this->print_render_attribute_string('carousel_nav_next_btn'); ?>><?php
				echo $nav_icons['next'];
			?></button>
			<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'end' ) {
				$this->get_slides_indicator( $settings );
			} ?>
		</div><?php

	}

	protected function get_dots() {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['pagination_dots'] ) ) return '';

		$slides_indicator = $settings['slides_indicator'];
		$slides_indicator_is_inline = $settings['slides_indicator_inline_w_dots'];
		$slides_indicator_placement = $settings['slides_indicator_inline_w_dots_placement'];
		$should_get_slides_indicator = $slides_indicator === 'yes' && $slides_indicator_is_inline === 'yes';

		if ( $should_get_slides_indicator && $slides_indicator_placement === 'start' ) {
			$this->get_slides_indicator( $settings );
		} ?>
		<div class="lqd-carousel-dots lqd-carousel-dots-empty flex"></div>
		<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'end' ) {
			$this->get_slides_indicator( $settings );
		}

	}

	protected function get_slides_indicator() {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['slides_indicator'] ) ) return '';

		$this->add_render_attribute( 'slides-indicator-wrapper', 'class', 'lqd-carousel-slides-ind flex items-center gap-20 text-14 lqd-carousel-slides-ind-'. esc_attr( $this->get_id() ) );

		if ( $settings['slides_indicator_tail'] === '' ) {
			$this->add_render_attribute( 'slides-indicator-wrapper', 'class', 'lqd-carousel-slides-ind-no-tail');
		}

		?><div <?php $this->print_render_attribute_string( 'slides-indicator-wrapper') ; ?>>
			<span class="lqd-carousel-slides-ind-current shrink-0"></span>
			<span class="lqd-carousel-slides-ind-progress-wrap flex items-center relative">
				<span class="lqd-carousel-slides-ind-progress-bar inline-block w-full h-full absolute top-0 start-0 lqd-transform origin-left transition-all"></span>
			</span>
			<span class="lqd-carousel-slides-ind-total shrink-0"></span>
		</div><?php

	}

	protected function get_cell_overlay( $cell ) {
		if (
			empty( $this->get_settings_for_display('cell_overlay_bg_background') ) &&
			empty( $this->get_settings_for_display('cell_overlay_bg_hover_background') ) &&
			empty( $this->get_settings_for_display('cell_overlay_bg_active_background') )
		) return;

		?>
		<span class="lqd-carousel-cell-overlay block w-full h-full absolute top-0 start-0 pointer-events-none transition-all"></span>
		<?php
	}

	/**
	 * Render the video background markup.
	 *
	 * @return void
	 */
	private function render_video_background( $item ) {
		if ( 'video' !== $item['background_background'] ) {
			return;
		}

		if ( ! $item['background_video_link'] ) {
			return;
		}

		$video_properties = Embed::get_video_properties( $item['background_video_link'] );

		$this->add_render_attribute(
			'background-video-container',
			'class',
			[
				'lqd-bg-video-container',
				'lqd-bg-video-container-' . $item['_id'],
				'w-full',
				'h-full',
				'absolute',
				'top-0',
				'start-0',
				'pointer-events-none',
			]
		);

		if ( ! $item['background_play_on_mobile'] ) {
			$this->add_render_attribute( 'background-video-container', 'class', 'elementor-hidden-phone' );
		}

		?><div <?php $this->print_render_attribute_string( 'background-video-container' ); ?>>
			<?php if ( $video_properties ) : ?>
				<div class="lqd-bg-video-embed absolute top-1/2 start-1/2 lqd-transform -translate-x-1/2 -translate-y-1/2 !max-w-none"></div>
				<?php
			else :
				$video_tag_attributes = 'autoplay muted playsinline';

				if ( $item['background_play_once'] !== 'yes' ) {
					$video_tag_attributes .= ' loop';
				}
				?>
				<video class="lqd-bg-video-hosted lqd-html5-video" <?php echo esc_attr( $video_tag_attributes ); ?>></video>
			<?php endif; ?>
		</div><?php
	}

	protected function get_cell_content( $item, $i ) {
		$settings = $this->get_active_settings();
		$title = $item['title'];
		$description = $item['description'];
		$title_tag = $settings['title_tag'];
		$classnames = [ 'lqd-v-slideshow-cell-content', 'lqd-carousel-cell-content', 'flex', 'grow', 'overflow-hidden', 'transition-effects' ];
		$attrs = [
			'class' => $classnames
		];
		$attrs_id = $this->get_repeater_setting_key( 'cell_content', 'cells', $i );

		$this->add_render_attribute( $attrs_id, $attrs );

		$this->get_cell_image( $item, $i );
		?>
		<div <?php $this->print_render_attribute_string( $attrs_id ) ?>>
			<div class="lqd-carousel-cell-text lqd-v-slideshow-text w-full relative z-1 transition-effects"><?php
				echo sprintf(
					'<%1$s class="lqd-carousel-cell-title lqd-v-slideshow-title m-0 transition-effects">%2$s</%1$s>',
					Utils::validate_html_tag( $title_tag ),
					$title,
				);
				echo sprintf(
					'<div class="lqd-carousel-cell-description lqd-v-slideshow-description m-0 transition-effects">%1$s</div>',
					$description,
				);
			?></div>
		</div>
		<?php
	}

	protected function get_cells_contents() {
		$settings = $this->get_settings_for_display();
		$cells = $settings['cells'];
		$cells_common_classnames = [ 'lqd-v-slideshow-cell', 'lqd-carousel-cell', 'lqd-filter-item', 'flex', 'shrink-0', 'grow-0', 'basis-auto', 'relative', 'transition-effects' ];

		foreach ( $cells as $i => $cell ) {

			$cell_attrs_id = $this->get_repeater_setting_key( 'cell', 'cells', $i );
			$cell_classnames = array_merge( $cells_common_classnames, [ 'elementor-repeater-item-' . esc_attr( $cell['_id'] ) . '' ] );

			$this->add_render_attribute( $cell_attrs_id, [
				'class' => $cell_classnames
			] );

		?>

		<div <?php $this->print_render_attribute_string( $cell_attrs_id ) ?>><?php
			$this->get_cell_content( $cell, $i );
		?></div>

		<?php }
	}

	protected function get_cell_image( $item, $i ) {
		$settings = $this->get_active_settings();
		$classnames = [ 'lqd-v-slideshow-image', 'h-full', 'absolute', 'top-0', 'z-0', 'overflow-hidden', 'pointer-events-none' ];
		$attrs = [
			'class' => $classnames
		];
		$attrs_id = $this->get_repeater_setting_key( 'cell_image', 'cells', $i );

		$this->add_render_attribute( $attrs_id, $attrs );

		?>
		<div <?php $this->print_render_attribute_string( $attrs_id ) ?>><?php
			$this->render_video_background( $item );
			$this->get_cell_overlay( $item );
		?></div>
		<?php
	}

	protected function render() {
		$settings = $this->get_settings_for_display();
		$items = $settings['items'];
		$slides_indicator = $settings['slides_indicator'];
		$slides_indicator_is_inline = ( $settings['nav_buttons'] === 'yes' && $settings['slides_indicator_inline_w_nav'] === 'yes' ) ||
			($settings['pagination_dots'] === 'yes' && $settings['slides_indicator_inline_w_dots'] === 'yes');

		?>
		<div class="lqd-carousel-viewport relative overflow-hidden">
			<div class="lqd-carousel-slider flex relative"><?php
				$this->get_cells_contents();
			?></div>
		</div>

		<?php
			$this->get_nav();
			$this->get_dots();
			if ( !$slides_indicator_is_inline ) {
				$this->get_slides_indicator();
			}
		?>

		<?php
	}

}
\Elementor\Plugin::instance()->widgets_manager->register( new LQD_Vertical_Slideshow() );